# btc64.py
# Generates fully automated 64-vector HDGL covenant script
# Saturates vectors safely with minimal r_dim + Ω embedding

MAX_VECTORS = 64
BLOCK_SIZE = 4  # 4 vectors per block
SAFE_ZERO = "0x0"

# Example r_dim & Ω per vector block (powers-of-2 scaled for safety)
# Only unique nonzero params are embedded
PARAMS = {
    1: ("0x3000", "0x812"),
    2: ("0x4000", "0x502"),
    3: ("0x5000", "0x310"),
    4: ("0x6000", "0x192"),
    5: ("0x7000", "0x119"),
    6: ("0x8000", "0x73"),
    7: ("0x9000", "0x45"),
    8: ("0xA000", "0x28"),
    # Blocks 9..16 will repeat last safe saturation
}

def generate_hdgl_covenant(pubkey="<YOUR_PUBLIC_KEY>"):
    script = [f"{pubkey} OP_CHECKSIGVERIFY\n"]

    for block_idx in range(1, MAX_VECTORS // BLOCK_SIZE + 1):
        # vectors: v0..v3
        v_line = " ".join([SAFE_ZERO]*BLOCK_SIZE) + " OP_HDGL_VEC_ADD OP_EQUALVERIFY"
        script.append(v_line)

        # indices for OP_ADD and OP_WITHINVERIFY
        idx_base = 100 + (block_idx-1)*BLOCK_SIZE
        add_line = f"{idx_base} {idx_base+1} OP_ADD {idx_base+2} OP_EQUALVERIFY"
        script.append(add_line)

        within_line = f"{idx_base+3} {idx_base+4} {idx_base+5} OP_WITHINVERIFY"
        script.append(within_line)

        # embed unique r_dim & Ω params if defined, else repeat last
        if block_idx in PARAMS:
            r_dim, omega = PARAMS[block_idx]
        else:
            # repeat last defined param
            last_block = max([k for k in PARAMS.keys() if k < block_idx])
            r_dim, omega = PARAMS[last_block]
        param_line = f"{r_dim} {omega} OP_HDGL_PARAM_ADD OP_EQUALVERIFY"
        script.append(param_line)

        script.append("")  # empty line for readability

    # final covenant check
    script.append("OP_CHECKTEMPLATEVERIFY")
    return "\n".join(script)

if __name__ == "__main__":
    covenant_script = generate_hdgl_covenant()
    print("=== Generated Compressed 64-Vector HDGL Covenant ===")
    print(covenant_script)
